/***************************************************************************
 *   Copyright (C) 2011 by Jeremy Burton   *
 *   jburton@39net-w04   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <math.h>
#include <QPainter>

#include <stdio.h>

#include "dom.h"
#include "shapeitem.h"

shapeitem::shapeitem() : baseitem()
{
  strokewidth=0.0;
  stroke=false;
  fill=true;
  fillcolor=Qt::black;
  scale=1.0;
  _extent=0.0;
}

////////////////////////////////////////////////////////////////////////////////

baseitem *shapeitem::dup()
{
  shapeitem *item = new shapeitem;
  *item = *this;
  return item;
}

////////////////////////////////////////////////////////////////////////////////

double shapeitem::extent()
{
  return _extent;
}

////////////////////////////////////////////////////////////////////////////////

QString shapeitem::describe()
{
  return QString("Shape: %1").arg(name);
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::update()
{
  CPointF p;
  double r;
  _extent=0.0;

  for (int i=0;i<points.size();i++)
  {
    p=points.at(i)*scale+offset;
    r=p.r();
    if (r>_extent) _extent=r;
  }
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::toDom(QDomDocument &doc,QDomElement &parent)
{
  QStringList sl;
  QDomElement aw = doc.createElement( "shapeitem" );

  //aw.setAttribute(QString("square"),square);

  baseitem::toDom_content(doc,aw);

  for (int i=0; i<points.size(); i++)
    sl << QString::number(points[i].x()) << QString::number(points[i].y());

  toDom_Text(doc,aw,"name",name);
  toDom_Text(doc,aw,"offsetx",offset.x());
  toDom_Text(doc,aw,"offsety",offset.y());
  toDom_Text(doc,aw,"scale",scale);
  toDom_Text(doc,aw,"points",sl.join(","));

  parent.appendChild(aw);
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::fromDom(QDomDocument &doc,QDomElement &element)
{
  baseitem::fromDom(doc,element);

  //square = element.attribute("square", "1").toInt() != 0;

  QDomNode n = element.firstChild();
  while(!n.isNull()) 
  {
    QDomElement e = n.toElement(); // try to convert the node to an element.
    if(!e.isNull())
    {
      if (e.tagName() == "name")
      {
        name = e.text();
      }
      else
      if (e.tagName() == "offsetx")
      {
        offset.setX(e.text().toDouble());
      }
      else
      if (e.tagName() == "offsety")
      {
        offset.setY(e.text().toDouble());
      }
      else
      if (e.tagName() == "scale")
      {
        scale = e.text().toDouble();
      }
      else
      if (e.tagName() == "points")
      {
	QStringList sl = e.text().split(",");
	for (int i=0; (i+1)<sl.size(); i+=2)
	  points << QPointF(sl[i].toDouble(),sl[i+1].toDouble());
      }
    }
    n = n.nextSibling();
  }
  update();
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::paint_handle_offset(QPainter &painter,const QPoint &pos)
{
  if (centrelock) return;

  QColor
    color(Qt::cyan);

  color.setAlpha(127);
  //move_handle = ctr;

  painter.setBrush(color);
  painter.setPen(Qt::NoPen);
  painter.drawRect(pos.x()-MOVE_HANDLE_SIZE/2,pos.y()-MOVE_HANDLE_SIZE/2,MOVE_HANDLE_SIZE,MOVE_HANDLE_SIZE);
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::paint_handles(QPainter &painter, double zoom)
{
  QPoint
    ctr(centre.x()*zoom,-centre.y()*zoom),
    ofs(offset.x()*zoom,-offset.y()*zoom),
    ctr1(0,0);

  move_handle = ctr;
  angle_handle.setPolar(ctr,_extent*zoom+ANGLE_HANDLE_EXTEND,angle);
  offset_handle = ctr+ofs;
  offset_handle.rotate(ctr,angle);

  painter.translate(ctr);
  painter.rotate(angle);
  paint_handle_move(painter,ctr1);
  paint_handle_angle2(painter,ctr1,_extent*zoom);
  paint_handle_offset(painter,ctr1+ofs);
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::paint(QPainter &painter, bool selected, double zoom)
{
  QPoint
    ctr(centre.x()*zoom,-centre.y()*zoom);
    //ofs(offset.x(),-offset.y());
  QPen
    pen1(Qt::SolidLine);
  QColor
    scolor(strokecolor),
    fcolor(fillcolor);

  if (!selected)
  {
    scolor.setAlpha(127);
    fcolor.setAlpha(127);
  }

  if (stroke)
  {
    pen1.setWidth(strokewidth*zoom);
    pen1.setJoinStyle(Qt::MiterJoin);
    pen1.setColor(scolor);
    painter.setPen(pen1);
  }
  else
    painter.setPen(Qt::NoPen);

  if (fill)
    painter.setBrush(fcolor);
  else
    painter.setBrush(Qt::NoBrush);

  painter.translate(ctr);
  painter.rotate(angle);
  painter.scale(zoom,-zoom);
  painter.translate(offset);
  painter.scale(scale,scale);

  painter.drawPolygon(points);
}

////////////////////////////////////////////////////////////////////////////////

void shapeitem::output(QPainter &painter)
{
  int
    dpix = painter.device()->logicalDpiX(),
    dpiy = painter.device()->logicalDpiY()/*,
    w = realtodev(width,dpix),
    h = realtodev(height,dpiy)*/;
  QPoint
    ctr(realtodev(centre.x(),dpix),realtodev(-centre.y(),dpiy));
  QPen
    pen1(strokecolor);

//printf("DPI=%d,%d\n",dpix,dpiy);

  if (stroke)
  {
    pen1.setWidth(realtodev(strokewidth,dpix));
    pen1.setJoinStyle(Qt::MiterJoin);
    painter.setPen(pen1);
  }
  else
    painter.setPen(Qt::NoPen);

  if (fill)
    painter.setBrush(fillcolor);
  else
    painter.setBrush(Qt::NoBrush);

  painter.translate(ctr);
  painter.rotate(angle);
  //painter.drawRect(-w/2.0,-h/2.0,w,h);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

